pub type T = f32;
const EPSILON : T = std::f32::EPSILON;
//const EPSILON : T = 0;
const ZERO : T = 0.0;
const ONE : T = 1.0;

#[derive(Clone)]
pub struct Vector3 {
    pub x: T,
    pub y: T,
    pub z: T,
}

impl From<(T,T,T)> for Vector3 {
    fn from(other: (T,T,T)) -> Vector3 {
        Vector3 { x: other.0, y: other.1, z: other.2 }
    }
}

impl std::ops::Add<&Vector3> for &Vector3 {
    type Output = Vector3;
    fn add(self, other: &Vector3) -> Vector3 {
        Vector3 { x: self.x + other.x, y: self.y + other.y, z: self.z + other.z }
    }
}

impl std::ops::Sub<&Vector3> for &Vector3 {
    type Output = Vector3;
    fn sub(self, other: &Vector3) -> Vector3 {
        Vector3 { x: self.x - other.x, y: self.y - other.y, z: self.z - other.z }
    }
}

impl std::ops::Mul<T> for &Vector3 {
    type Output = Vector3;
    fn mul(self, other: T) -> Vector3 {
        Vector3 { x: self.x * other, y: self.y * other, z: self.z * other }
    }
}

fn dot(a: &Vector3, b: &Vector3) -> T {
    a.x*b.x + a.y*b.y + a.z*b.z
}

fn cross(a: &Vector3, b: &Vector3) -> Vector3 {
    Vector3 {
        x: a.y*b.z - a.z*b.y,
        y: -(a.x*b.z - a.z*b.x),
        z: a.x*b.y - a.y*b.x,
    }
}

fn sq_dist_point_segment(a: &Vector3, b: &Vector3, c: &Vector3) -> T {
    let ab = b- a;
    let ac = c- a;
    let bc = c - b;
    
    let e = dot(&ac, &ab);
    if e <= ZERO {
        return dot(&ac, &ac);
    }
    
    let f = dot(&ab, &ab);
    if e >= f {
        return dot(&bc, &bc);
    }

    dot(&ac, &ac) - e*e/f
}

fn test_sphere_capsule(sphere_center: &Vector3, sphere_rad: T, capsule_a: &Vector3, capsule_b: &Vector3, capsule_rad: T) -> bool {
    let dist2 = sq_dist_point_segment(&capsule_a, &capsule_b, &sphere_center);
    let rad = sphere_rad + capsule_rad;
    dist2 <= rad*rad
}

fn test_sphere_sphere(sphere_center_a: &Vector3, sphere_rad_a: T, sphere_center_b: &Vector3, sphere_rad_b: T) -> bool {
    let diff = sphere_center_b - sphere_center_a;
    let dist2 = dot(&diff, &diff);
    let rad = sphere_rad_a + sphere_rad_b;
    dist2 <= rad*rad
}

fn clamp(value: T, min: T, max: T) -> T {
    return if value <= min {
        min
    } else if value >= max {
        max 
    } else {
        value
    }
}

fn clamp01(value: T) -> T {
    clamp(value, ZERO, ONE)
}

fn closest_pt_segment_segment(p1: &Vector3, q1: &Vector3, p2: &Vector3, q2: &Vector3) -> (T, T, T, Vector3, Vector3) {
    let mut s = ZERO;
    let mut t = ZERO;

    let d1 = q1 - p1;
    let d2 = q2 - p2;
    let r = p1 - p2;
    let a = dot(&d1, &d1);
    let e = dot(&d2, &d2);
    let f = dot(&d2, &r);

    if a <= EPSILON && e <= EPSILON {
        let result = dot(&(p1 - p2), &(p1 - p2));
        return (result, s, t, p1.clone(), p2.clone());
    } 

    if a <= EPSILON {
        t = clamp01(f / e);
    } else {
        let c = dot(&d1, &r);
        if e <= EPSILON {
            s = clamp01(-c / a);
        } else {
            let b = dot(&d1, &d2);
            let denom = a*e - b*b;

            if denom != ZERO {
                s = clamp01((b*f - c*e) / denom);
            } else {
                t = (b*s + f) / e;
                if t < ZERO {
                    t = ZERO;
                    s = clamp01(-c / a);
                } else if t > ONE {
                    t = ONE;
                    s = clamp01((b-c) / a);
                }
            }
        }
    }

    let c1 = p1 + &(&d1*s);
    let c2 = p2 + &(&d2*t);
    let result = dot(&(&c1 - &c2), &(&c1 - &c2));
    
    (result, s, t, c1, c2)
}

fn test_capsule_capsule(capsule0_a: &Vector3, capsule0_b: &Vector3, capsule0_rad: T, 
    capsule1_a: &Vector3, capsule1_b: &Vector3, capsule1_rad: T) -> bool
{
    let (d2, _s, _t, _c1, _c2) = 
        closest_pt_segment_segment(&capsule0_a, &capsule0_b, &capsule1_a, &capsule1_b);
    let rad = capsule0_rad + capsule1_rad;
    d2 <= rad*rad
}


fn test_segment_triangle(p: &Vector3, q: &Vector3, a: &Vector3, b: &Vector3, c: &Vector3) -> (bool, T, T, T, T) {
    let false_result = (false, ZERO, ZERO, ZERO, ZERO);

    let ab = b - &a;
    let ac = c - &a;
    let qp = p - &q;

    let n = cross(&ab, &ac);
    let d = dot(&qp, &n);

    if d <= ZERO {
        return false_result;
    }

    let ap = p - &a;
    let t = dot(&ap, &n);
    if t <= ZERO || t > d {
        return false_result;
    }

    let e = cross(&qp, &ap);
    let v = dot(&ac, &e);
    if v < ZERO || v > d {
        return false_result;
    }

    let w = -dot(&ab, &e);
    if w < ZERO || v + w > d {
        return false_result;
    }

    let inv_d = ONE / d;
    let t = t * inv_d;
    let v = v * inv_d;
    let w = w * inv_d;
    let u = ONE - v - w;

    (true, u, v, w, t)
}

pub fn run_test(
    spheres: &Vec<((T,T,T), T)>, 
    capsules: &Vec<((T,T,T), (T,T,T), T)>,
    segments: &Vec<((T,T,T), (T,T,T))>,
    triangles: &Vec<((T,T,T), (T,T,T), (T,T,T))>
) -> (i32, u128) 
{
    let mut num_overlaps = 0;

    let spheres : Vec<(Vector3, T)> = spheres.iter().map(|s| (s.0.into(), s.1)).collect();
    let capsules : Vec<(Vector3, Vector3, T)> = capsules.iter().map(|c| (c.0.into(), c.1.into(), c.2)).collect();
    let segments : Vec<(Vector3, Vector3)> = segments.iter().map(|s| (s.0.into(), s.1.into())).collect();
    let triangles : Vec<(Vector3, Vector3, Vector3)> = triangles.iter().map(|t| (t.0.into(), t.1.into(), t.2.into())).collect();

    let start = std::time::Instant::now();

    // All sphere-sphere intersections
    for (i, s0) in spheres.iter().enumerate() {
        for s1 in &spheres[i+1..] {
            let overlap = test_sphere_sphere(&s0.0, s0.1, &s1.0, s1.1);
            if overlap {
                num_overlaps += 1;
            }
        }
    }
    //println!("Sphere: {} -> {}", num_overlaps, num_overlaps);

    // All sphere-capsule intersections
    //let last = num_overlaps;
    for s in spheres.iter() {
        for c in capsules.iter() {
            let overlap = test_sphere_capsule(&s.0, s.1, &c.0, &c.1, c.2);
            if overlap {
                num_overlaps += 1;
            }
        }
    }
    //println!("SphereCapsule: {} -> {}", num_overlaps - last, num_overlaps);

    // All capsule-capsule intersections
    //let last = num_overlaps;
    for (i, c0) in capsules.iter().enumerate() {
        for c1 in &capsules[i+1..] {
            let overlap = test_capsule_capsule(&c0.0, &c0.1, c0.2, &c1.0, &c1.1, c1.2);
            if overlap {
                num_overlaps += 1;
            }
        }
    }
    //println!("CapsuleCapsule: {} -> {}", num_overlaps - last, num_overlaps);

    // All segment-triangle intersections
    //let last = num_overlaps;
    for s in segments.iter() {
        for t in triangles.iter() {
            let (overlap, _, _, _, _) = test_segment_triangle(&s.0, &s.1, &t.0, &t.1, &t.2);
            if overlap {
                num_overlaps += 1;
            }
        }
    }
    //println!("SegmentTriangle: {} -> {}", num_overlaps - last, num_overlaps);

    let end = std::time::Instant::now();
    let elapsed = (end - start).as_millis();

    (num_overlaps, elapsed)
}